;;; -*- Mode:Common-Lisp; Package:TV; Base:10; Fonts:(CPTFONT HL10B HL12BI HL12B CPTFONTB)1; *Patch-File: T -*-

;1;;                     RESTRICTED RIGHTS LEGEND          *
;1;; Use, duplication, or disclosure by the Government is subject to*
;1;; restrictions as set forth in subdivision (b)(3)(ii) of the Rights in*
;1;; Technical Data and Computer Software clause at 52.227-7013.*
;1;;                   TEXAS INSTRUMENTS INCORPORATED.*
;1;;                            P.O. BOX 149149*
;1;;                         AUSTIN, TEXAS 78714-9149*
;1;;                             MS 2151*
;1;; Copyright (C) 1986,1987, 1988, 1989 Texas Instruments Incorporated. All rights reserved.*

;1; We need graphs to be able to add them as perspectives.*

(DEFVAR 4*default-call-graph-depth** 2
"2The default depth for call graphs.*")

(DEFTYPE 4real-compiled-function* ()
"2This type is true for any sort of compiled function, be it a ucode entry
 or a normal compiled function.*"
  '(OR compiled-function microcode-function))

(DEFUN 4the-fef* (x)
"2Given something tries to coerce it into a fef.  If it is a symbol then it tries
 to do clever things to get a fef that denotes it, e.g. its macro function.
 If it's something it knows about, but can't get a fef for it, then it just
 returns X.  If it doesn't know about its type at all then it barfs.
 The types it knows about are:  Symbols, Compiled functions, ucode functions,
 clos methods, clos generic functions and grapher vertices.*"
  (TYPECASE x
    (symbol (IF (FBOUNDP x)
	        (LET ((DEF (fdefinition-safe x t)))
		    (IF (CONSP def)
		        (IF (TYPEP (REST def) 'real-compiled-function)
			    (REST def)
			    x)
			(IF (TYPEP def 'real-compiled-function)
			    def
			    x)))
	       x))
    (compiled-function x)
    (microcode-function x)
    (any-type-of-clos-method (get-fef-from-object x))
    (any-type-of-clos-gf (get-fef-from-object x))
    (vertex x)
    (CONS (IF (EQUAL (FIRST x) 'MACRO)
	      (REST x)
	      x))
    (otherwise (OR (fdefinition-safe x t)
		   (FERROR nil "3~S is not a function.*" x)))))

(DEFVAR 4*selected-filters** nil
"2The list of filter functions currently selected by the user.  This is a useful
 variable to set in your init file so that you prune call graphs quite a bit
 by default.*")

(DEFUN 4maybe-filter* (filters fun as child-of window)
"2Is passed a list of filter functions, function (probably a fef) a comparison
 function and the function of which this is a child.  The comparison function
 is the function that was recorded if filters were selected by mousing on a
 function.  This allows filters such as \"same package as this function\". 
 If any of the functions in the list of funtions returns true for the function
 when funcalled as (filter-fun fun as child-of) then the function is filtered
 out.*"
  (DECLARE (optimize (safety 0)))
  (IF filters
      (IF (FUNCALL (FIRST filters) fun as child-of window)
	  nil
	  (maybe-filter (REST filters) fun as child-of window))
      fun))

(DEFVAR 4*as-function** nil
"2A place where a comparison function is stored.  This is the function on which
 the user last right clicked and put up the filter menu.  This allows filters
 like \"same package as this function\" to work.*")

(DEFUN 4filter-called-functions* (functions child-of window)
"2Given the functions (Functions) that are called by Child-Of, filter out all of
 those that should be filtered out according to the user's currently selected
 set of filter functions.*"
  (DECLARE (SPECIAL *as-function* *selected-filters*))
  (IF *selected-filters*
     (REMOVE-IF-NOT
       #'(lambda (fun)
	   (maybe-filter *selected-filters* fun
			 (IF (BOUNDP '*as-function*)
			     *as-function*
			     nil)
			 child-of window))
         functions)
     functions))

(DEFPARAMETER 4*function-details-cache** (MAKE-HASH-TABLE :test #'EQUAL)
"2A hash table that maps functions to the ivars-and-messages-in-method
 data for that function.  This is used to speed up call graphing for non-first
 time graphs.*")

(DEFUN 4maybe-cache-function-values* (name fef function)
"2Looks in the cache for the function Fef, whose name is Name and if it
 finds values then returns these, otherwise it calls Function on Fef and caches
 the results returning the values of the function as its values.*" 
  (LET ((cache (GETHASH name *function-details-cache*)))
      (IF cache
	  (VALUES-LIST cache)
          (LET ((results
		  (MULTIPLE-VALUE-LIST (FUNCALL function fef))))
	      (SETF (GETHASH name *function-details-cache*)
		    results)
	      (VALUES-LIST results)))))

(DEFUN 4caching-ivars-and-messages-in-method* (fef)
"2Calls ivars-and-messages-in-method on Fef and caches the results as
 appropriate.  Otherwise it's just like a call to
 ivars-and-messages-in-method.*"
  (maybe-cache-function-values
    (FUNCTION-NAME fef) fef 'ivars-and-messages-in-method))

(DEFUN 4functions-called-from-interpretted-definition-1* (FUNCTION)
"2Given an interpretted function it grovels over the function and records all
 of the functions and generic functions called.  These are collected in the
 specials *all-functions-called* *all-gfs-called*.*"
  (DECLARE (SPECIAL *all-functions-called* *all-gfs-called*))
  (DECLARE (optimize (safety 0)))
  (TYPECASE function
    (CONS (functions-called-from-interpretted-definition-1 (FIRST function))
	  (functions-called-from-interpretted-definition-1 (REST function)))
    (symbol (IF (FBOUNDP function)
		(IF (generic-function-p-safe function)
		    (PUSHNEW function *all-gfs-called*)
		    (PUSHNEW function *all-functions-called*))
		nil))
    (otherwise nil)))

(DEFUN 4functions-called-from-interpretted-definition* (FUNCTION)
"2Given an interpretted function or the name of an interpretted function it
 returns values as if ivars-and-messages-in-method had been called for
 a fef, i.e. it returns sundry null values for the things it can't figure out
 and values for the functions called and the generic functions called.*"
  (DECLARE (VALUES ignore ignore ignore functions-called
		    generic-functions-called ignore ignore ignore ignore ignore))
  (DECLARE (optimize (safety 0)))
  (LET ((*all-functions-called* nil)
	(*all-gfs-called* nil))
       (DECLARE (SPECIAL *all-functions-called* *all-gfs-called*))
       (functions-called-from-interpretted-definition-1
	 (IF (AND (SYMBOLP function) (FBOUNDP function))
	     (SYMBOL-FUNCTION function)
	     function))
       (VALUES nil nil nil (NREVERSE *all-functions-called*)
	        (NREVERSE *all-gfs-called*) nil nil nil nil nil)))

(DEFUN 4caching-functions-called-from-interpretted-definition* (name)
"2Just like Functions-Called-From-Interpretted-Definition, only the results
 of calling this function are cached.  It returns, amongst a number of null
 values the functions called by Name and the generic functions called by Name.*"
  (DECLARE (VALUES ignore ignore ignore functions-called
		    generic-functions-called ignore ignore ignore ignore ignore))
  (maybe-cache-function-values
    name name
    #'functions-called-from-interpretted-definition))

(DEFUN 4get-function-call-data* (FUNCTION)
"2Is called with a function, which can be either a compiled function, an
 interpretted function or the name of an interpretted fucntion.  It returns
 three values:  the non-generic functions called by Function (including the
 methods related to Function if Function is a generic function), the generic
 functions called by Function and the macros that were expanded in processing
 Function.*"
  (DECLARE (VALUES functions-called-including-methods-for-gfs
		   generic-functions-called macros-expanded))
  (MULTIPLE-VALUE-BIND
    (referenced-ivars referenced-keywords problem referenced-functions
     referenced-generic-functions args returned-values locals
     specials-referenced specials-bound)
      (IF (COMPILED-FUNCTION-P function)
	  (caching-ivars-and-messages-in-method function)
	  (caching-functions-called-from-interpretted-definition function))
    (IGNORE referenced-ivars referenced-keywords problem
     args returned-values locals specials-referenced specials-bound)
    (VALUES (APPEND (IF (generic-function-p-safe function)
			 (generic-function-methods-safe
			   (function-generic-function-safe function))
			 nil)
		     referenced-functions)
	    referenced-generic-functions
	    (MAPCAR #'ucl::first-if-list
		     (IF (AND (FUNCTIONP function)
			       (sys:get-debug-info-struct function))
			 (GETF (sys:dbis-plist
				 (sys:get-debug-info-struct function))
			       :macros-expanded)
			 nil)))))

(DEFUN 4functions-called* (by-function)
"2Returns two values: the functions called by By-Function and a flag which is
 true if By-Function is recursive, i.e. By-Function is in the first value.
 The functions returned have been passed through the function filter, so only
 those functions not excluded by the filter will be returned.  The functions
 returned are returned as fefs where possible.*"
  (DECLARE (VALUES functions-called by-function-is-recursive-p)
	   (SPECIAL *graph-window*))
  (LET ((fef (the-fef by-function)))
       (MULTIPLE-VALUE-BIND
	 (referenced-functions referenced-generic-functions macros)
	   (get-function-call-data fef)
	 (LET ((kids (MAPCAR #'the-fef
			 (APPEND referenced-functions
				  referenced-generic-functions))))
	      (VALUES (filter-called-functions
			(APPEND
			  (REMOVE fef kids)
			  (MAPCAR #'the-fef macros))
			fef *graph-window*)
		      (MEMBER fef kids))))))

(DEFVAR 4*graph-calls-print-package** "3User*"
"2The default package to bind to during printing when doing a graph calls.*")
(DEFVAR 4*printing-bindings** nil
"2A list of (*special* value) pairs.  bindings to these specials are made
 with these values whilst printing functions in the call grapher.  e.g.
 `((*print-case* :capitalize) (*package* ,(find-package \"TV\")))*")

(DEFUN 4function-calls-others* (name)
"2Is true if the function named by Name calls any other functions.  This allows
 non-terminal functions to be labeled as such.*" 
  (LET ((fef (the-fef name)))
       (MULTIPLE-VALUE-BIND
	 (referenced-functions referenced-generic-functions macros)
	       (get-function-call-data fef)
	 (OR referenced-functions referenced-generic-functions macros))))

(DEFUN 4more-kids-indicator* (kids)
"2Returns a  or a  if there are kids, otherwise the null string.*"
  (LET ((window (IF (TYPEP self 'vertex)
		      (SEND self :window)
		      (IF (TYPEP self 'basic-x-y-scrolling-window)
			 self
			 nil))))
      (IF (AND window kids)
	 (IF (AND (mx-p)
		  (OR (NOT (TYPEP self 'vertex))
		      (NOT (EQUAL fonts:cptfont (SEND self :font)))))
	     (IF (EQUAL :vertical (SEND window :orientation)) "" "3->*")
	     (IF (EQUAL :vertical (SEND window :orientation)) "3*" "3*"))
	  "")))

(DEFVAR 4*fef-print-functions**
	`(,#'(lambda (name)
	       (IF (CONSP name) (FORMAT t "3~*" name t) (FORMAT t "3~S*" name)))
	  ,#'(lambda (name fef)
	      (PRINC (more-kids-indicator
		       (IF (AND (TYPEP fef 'vertex-node)
				 (vertex-node-vertex fef))
			  (NOT (SEND (vertex-node-vertex fef) :child-edges))
			  (function-calls-others name))))))
"2Functions to call during the displaying of functions in the call grapher.
 the functions are called in sequence in the list each with the NAME of the
 function being displayed and optionally the vertex node.  These functions
 should print what they think is important about the function to
 *standard-output*.  An example of such a function that puts an arrow at
 the end of a display to indicate that there are children could be as follows:
  #'(lambda (name)
     (if (Function-Calls-Others name)*
	2 (if (equal :Vertical (send (send self :Window):Orientation))*
	2     (format t \"\")*
	2     (format t \"\"))*
	2 nil))*")

(DEFUN 4simple-print-fef* (fef)
"2A simple print function to print fefs.  Fef is actually either a fef or the
 name of a function.  It returns a string that will be displayed for the
 function.  It does this by calling the functions in *Fef-Print-Functions*.*"
  (LET ((*package*
	  (FIND-PACKAGE
	    (STRING-UPCASE
	      (STRING (IF (TYPEP self 'stand-alone-call-grapher)
			 (SEND self :print-package)
			 *graph-calls-print-package*))))))
       (PROGV (MAPCAR #'FIRST *printing-bindings*)
	      (MAPCAR #'SECOND *printing-bindings*)
         (WITH-OUTPUT-TO-STRING (*standard-output*)
	   (LET ((name (FUNCTION-NAME (coerce-to-node fef))))
	       (LOOP for fn in *fef-print-functions* do
		     (IF (> (LENGTH (ARGLIST fn)) 1)
			 (FUNCALL fn name fef)
			 (FUNCALL fn name))))))))

;1-------------------------------------------------------------------------------*

(DEFFLAVOR 4function-vertex* () (boxed-filled-vertex)
  (:default-init-plist :box-myself-p nil :fill-myself-p nil)
  (:documentation "3The flavor of grapher vertex to use for functions.*"))

(DEFUN 4interpretted-function-p* (FUNCTION)
"2Is true if Function is an interpretted function.*"
  (EQUAL (the-fef function) function))

(DEFVAR 4*function-vertex-boxing-selections**
	`((,#'MACRO-FUNCTION (t nil))
	  (,#'interpretted-function-p (nil t))
	  (,#'(lambda (FUNCTION) (IGNORE function) t) (nil nil)))
"2A list of pairs, each element of which is of the form:
  (<predicate> (<fill-p> <box-p> <dashed-p>))
 The predicate functions are called successively on functions until one of them
 is true.  When a match is found them the second value is used to set the
 filledness, the boxedness and the dashedness of the vertex on the graph.*")

(DEFMETHOD 4(function-vertex :after :init*) (IGNORE)
"2Makes sure that each vertex in the graph is set up so that it's box/fill
 status is right.  It does this by searching though the list
 *Function-Vertex-Boxing-Selections* until it gets a match and using the
 values it finds there.*" 
  (LET ((fn (FUNCTION-NAME (coerce-to-node item))))
       (LOOP for (FUNCTION (FILL box dash))
	     in *function-vertex-boxing-selections*
	     when (FUNCALL function fn)
	     do (PROGN (SET-IN-INSTANCE self 'dashed-p dash)
		       (SET-IN-INSTANCE self 'fill-myself-p fill)
		       (SET-IN-INSTANCE self 'box-myself-p box)
		       (RETURN nil)))))

;1-------------------------------------------------------------------------------*

(DEFMETHOD 4(function-vertex :print-self*) (STREAM &rest ignore)
"2Prints out a function vertex.*"
  (CATCH-ERROR
    (FORMAT stream "3#<Function Vertex ~A>*"
	    (FUNCTION-NAME (coerce-to-node item)))
    nil))

(DEFVAR 4*graph-calls-item-type-alist**
  '((:vertex edit-graphed-function
	     "3L: Edit this Function; M: Drag object; Sh-M/M2: Overview; R: Menu of operations*"
	     ("3Graph this Node*" :value maybe-graph-function :documentation
	      "3Start a new grapher whose top node is this function.*"
	     )
	     ("3Extend this Node*" :value grapher-entend-graph
	      :documentation "3Extend the graph below this node.*"
	     )
	     ("3Inspect*" :value graph-calls-inspect :documentation
	      "3Inspect this function.*"
	     )
	     ("3Edit*" :value edit-graphed-function :documentation
	      "3Edit the source code definition of this function.*"
	     )
	     ("3Filter*" :value select-grapher-filters :documentation
	      "3Select filtration for what functions to show on the graph.*"
	     )
	     ("3Variables*" :value set-graph-calls-variables :documentation
	      "3Set sundry variables for the call tree grapher.*")))

"2An item type alist for call tree grapher.  To add new right button menu
entries simply add the relevant menu item to the list after the third element of
the list for the relevant item type.*")

(DEFUN 4graph-calls-inspect* (FUNCTION)
"2Inspects a fucntion.  This gets called from the call grapher.*"
  (INSPECT function))

(DEFUN 4grapher-entend-graph* (FUNCTION window vertex)
"2Extends the graph for the current node.*"
  (IGNORE vertex)
  (extend-graph-for function window))

;1************* --------------------------------------------------*
;1 TAC 07-25-89 - moved up here from a lower position in this file *

(DEFVAR 4*graph-calls-orientation** :horizontal
"2The default orientation for the call grapher.*")

(DEFVAR 4*graph-calls-plot-trees-p** nil
"2The default for whether a new call grapher should plot trees or graphs.*")

(DEFVAR 4*grapher-selection-mode** :existing
"2The default for the way that new graphers are selected.*")
;1---------------------------------------------------------------*

(DEFFLAVOR 4stand-alone-call-grapher*
	   ((depth			*default-call-graph-depth*)
	    (print-package		*graph-calls-print-package*)
	    (orientation		*graph-calls-orientation*)
	    (tree-p			*graph-calls-plot-trees-p*)
	    (grapher-selection-mode	*grapher-selection-mode*)
	   )
	   (stand-alone-grapher)
  :gettable-instance-variables
  :settable-instance-variables
  :initable-instance-variables
  (:documentation
    "3A flavor of stand alone grapher that knows about how to plot call graphs.*"))

(DEFVAR 4*call-grapher-mouse-r-menu-items**
  '(("3Variables*" :value :set-graph-calls-variables :documentation
     "3Set sundry variables for the call tree grapher.*")
    ("3Filter*" :value :select-grapher-filters :documentation
     "3Select filtration for what functions to show on the graph.*")
    ("3Flush Cache*" :value :flush-function-details-cache
     :documentation "3Flush the functions called cache.*"))
"2The menu items that are used for the menu that gets popped up by right
 clicking on the background of the call grapher.*")

(DEFMETHOD 4(stand-alone-call-grapher :flush-function-details-cache*) (x y)
"2Flushes the cache of functions called by other functions.*"
  (IGNORE x y)
  (CLRHASH *function-details-cache*))

(DEFMETHOD 4(stand-alone-call-grapher :set-graph-calls-variables*) (x y)
"2Sets up sundry instance variables of the call grapher for use later on by
 popping up a menu.*"
  (IGNORE x y)
  (assign-using-menu
    ((depth "3Depth*" :fixnum)
     (print-package "3Package*" :string-or-nil)
     (orientation "3Orientation*" :assoc
		  (LIST (CONS "3Horizontal*" :horizontal)
		       (CONS "3Vertical*"   :vertical)))
     (tree-p "3Tree or Graph plot*" :assoc
	       (LIST (CONS "3Tree*" t) (CONS "3Graph*" nil)))
     (grapher-selection-mode "3Grapher selection mode.*" :assoc
	(LIST (CONS "3Use an existing one*" :existing)
	     (CONS "3Create a new one*" :new)
	     (CONS "3Ask*" :ask))))
    :label "3Set a grapher variables.*")
  (SEND self :set-graph-behaviour-changed t)
  (SETQ print-package (STRING-UPCASE (STRING print-package))))

;1 TAC 07-26-89 - moved up to here from a lower position in this file*
(DEFVAR 4*filter-items**
	'(("3Same Package*" :value same-package :documentation
	   "3Only those functions in the same package as selected function*")
	  ("3Not Sys:*" :value not-sys-functions :documentation
	   "3Not the functions in the Sys: package.*")
	  ("3No macros*" :value no-macros :documentation
	   "3Only those functions which are not (macro-function x).*")
	  ("3No Special Forms*" :value no-special-forms :documentation
	   "3Only those functions which are not special-form-p.*")
	  ("3Not called by Sys:*" :value not-called-by-sys-functions :documentation
	   "3Not the functions called by functions in the Sys: package.*")
	  ("3Not called by Lisp:*" :value not-called-by-lisp-functions :documentation
	   "3Not the functions called by functions in the Lisp: package.*")
	  ("3Not called by Macros*" :value not-called-by-macros :documentation
	   "3Filter out functions that are called by macros, but not the macros themselves.*")
	  ("3Not called by Special Forms*" :value not-called-by-special-forms
	   :documentation
	   "3Filter out functions that are called by special forms, but not the special forms themselves.*")
	  ("3Not called by Interpretted Functions*" :value
	   not-called-by-interpretted-functions :documentation
	   "3Filter out functions that are called by interpretted functions, but not the interpretted functions themselves.*")
	  ("3Not % Functions*" :value not-%-functions :documentation
	   "3Not any functions that have a \"%\" in their names.*")
	  ("3Not * Functions*" :value not-*-functions :documentation
	   "3Not any functions that have a \"*\" in their names.*")
	  ("3Only methods below generic functions*" :value
	   only-methods-below-generic-functions :documentation
	   "3Only the methods for generic functions, not the functions called by the generic function itself during method selection.*")
	  ("3Not methods below generic functions*" :value
	   not-methods-below-generic-functions :documentation
	   "3Not the methods for generic functions, but the functions called by the generic function itself during method selection.*")
	  ("3Same source file*" :value same-source-file :documentation
	   "3Only those functions that are in the same source file as the selected function.*"))
  "2This is a list of menu items that is called when the user wants to select some
 call grapher filter functions.  The items in the list are normal menu items,
 the value of the :value element must be a function with the following contract:
   arglist = (function as-function child-of)
   where Function is the function that we're trying to filter,
   As-Function is a function with which we may be comparing Function.
   This may be nil if there is no such function and is set when the
   user selects filters by right buttoning on a function and filtering
   from the menu or explicitly setting tv:*As-Function*.
   Child-Of is the parent node that called this function, if known (nil for the root node).
   Returned value = true if the Function arg is to be filtered out, nil otherwise.
 The function cannot guarantee that any of its args will be functions.  They
 should always call coerce-to-node if in doubt to make sure that it isn't a
 grapher internal data structure.  This may result in either function or a
 function name, so the filter function should do the righ thing with either. 
 Any of the functions in this list already will serve as examples.*")

(DEFMETHOD 4(stand-alone-call-grapher :select-grapher-filters*) (x y)
"2Pops up a menu of available filter functions and gets the user to select some.
 These are used in future to remove functions from the call graph.*"
  (IGNORE x y)
  (MULTIPLE-VALUE-BIND (choices chose-p)
      (w:multiple-menu-choose *filter-items*
	:label "3Filter Graphing*"
	:highlighted-items
	(REMOVE-IF-NOT
	  #'(lambda (x) (MEMBER (THIRD x) *selected-filters*))
	  *filter-items*))
      (IF chose-p
	  (PROGN (SETQ *selected-filters* choices)
		 (SEND self :set-graph-behaviour-changed t))
	  nil)))

(DEFMETHOD 4(stand-alone-call-grapher :mouse-r*) (x y)
"2Is called when the user right clicks on the background of the call grapher.*"
  (IGNORE x y)
  (LET ((choice (w:menu-choose *call-grapher-mouse-r-menu-items*)))
      (IF choice
	 (SEND self choice x y)
	 nil)))

(DEFMETHOD 4(stand-alone-call-grapher :mouse-r-who-line-doc*) ()
"2Documents the background right click method in the who line.*"
  '(:mouse-r-1 "3Menu of operations on the call grapher as a whole*"))

(DEFUN 4make-call-grapher-window* (&rest inits)
"2Makes a stand-alone call grapher window.*"
  (APPLY #'MAKE-INSTANCE 'stand-alone-call-grapher
	  :edges (offsetting-stand-alone-grapher-edges
		    'stand-alone-call-graphers)
	  inits))

;1; A resource of call graphers.*
(DEFWINDOW-RESOURCE 4stand-alone-call-graphers* ()
  :initial-copies 0
  :constructor (make-call-grapher-window))

;1(clear-resource 'Stand-Alone-Call-Graphers)*

(DEFUN 4graph-calls* (FUNCTION &optional (on :from-resource) (label nil))
"2Graphs the functions called by Function on grapher window On (when defaulted
 it allocates a new window from a resource).  Label is printed at the bottom
 of the window.  Function can be either the name of a function or a Fef.  If
 the display gets too complicated then it can be filtered by sundry filter
 functions.  These can be selected either by using the right button menus in
 grapher windows or by calling the function tv:Select-Grapher-Filters.  In
 your login file you can set the variable tv:*Selected-Filters* to a list of
 the names of the filter functions that you want.  For information on writing
 your own filter functions please look at the documentation/definition of
 tv:*Filter-Items*.*"
  (LET ((fef (the-fef function)))
      (IF (AND (TYPEP function 'vertex)
	       (SEND function :child-edges))
	 fef
	 (LET ((window
		(plot-a-graph
		  fef
		  :on on
		  :child-function 'functions-called
		  :print-function 'simple-print-fef
		  :vertex-flavor 'function-vertex
		  :item-type-alist *graph-calls-item-type-alist*
		  :depth (IF (TYPEP on 'stand-alone-call-grapher)
			     (SEND on :depth)
			     *default-call-graph-depth*)
		  :orientation (IF (TYPEP on 'stand-alone-call-grapher)
				  (SEND on :orientation)
				  *graph-calls-orientation*)
		  :tree-p (IF (TYPEP on 'stand-alone-call-grapher)
			      (SEND on :tree-p)
			      *graph-calls-plot-trees-p*)
		  :resource 'stand-alone-call-graphers
		  :dont-graph-if-already-graphing-this-node t
		  :label (OR label
			     (FORMAT nil "3~S's call graph.*"
				      (FUNCTION-NAME
					(FUNCTION-NAME fef)))))))
	     window))))

(DEFUN 4extend-graph-for* (FUNCTION &optional (on :from-resource))
"2Extends the graph on window On below the function Function.*"
  (LET ((*default-call-graph-depth* 1))
      (graph-calls (find-vertex-for (the-fef function) on) on :default)))

(DEFUN 4get-a-grapher-window*
       (&optional (how :existing) (resource 'stand-alone-graphers))
"2Gets a grapher window.  How can be :Existing, :New or :Ask.  This will cause
 either an existing window to be used, a new one to be allocated or you to
 be prompted about what to do.  New windows are allocated out of the resource
 Resource.*"
  (LET ((result nil))
       (MAP-RESOURCE #'(lambda (obj used-p ignore)
			 (IF used-p (PUSH obj result)))
		     resource)
       (LET ((window
	      (CASE how
		(:existing
		  (IF result
		      (LOOP for win in result
			    when (NOT (SEND win :exposed-p))
			    return win
			    finally (RETURN (ucl::first-if-list win)))
		      (ALLOCATE-RESOURCE resource)))
		(:new (ALLOCATE-RESOURCE resource))
		(:ask
		 (IF (AND result (mouse-y-or-n-p "3Use existing window?*"))
		     (get-a-grapher-window :existing resource)
		     (get-a-grapher-window :new resource)))
		(otherwise (FERROR nil "3Unknown selection type ~S.*" how)))))
	   window)))

;1; Make sure we can select call graphers.*

(add-system-key
  #\G '(find-window-of-flavor 'stand-alone-grapher selected-window)
  "2Grapher Window.*" nil)

;1-------------------------------------------------------------------------------*

(DEFUN 4inspect-graph-function-p* (name &optional (thing nil))
"2Is true if Name names a function which we can graph.  Thing is ignored.*"
  (IGNORE thing)
  (OR (AND (SYMBOLP name) (FBOUNDP name)
	   (OR (COMPILED-FUNCTION-P (SYMBOL-FUNCTION name))
	       (CONSP (SYMBOL-FUNCTION name))))
      (COMPILED-FUNCTION-P name)))

(DEFUN 4inspect-graph-function* (FUNCTION &optional (thing nil))
"2Graphs a function.  This is called from the inspector and such-like.  Thing
 is ignored.  A background process is spun off to do this so that selection
 problems are avoided.*"
  (IGNORE thing)
  (PROCESS-RUN-FUNCTION
    '(:name "3Expose Grapher*" :priority -1)
    #'(lambda (FUNCTION expose-p)
	(LET ((*force-exposure* expose-p))
	    (DECLARE (SPECIAL *force-exposure*))
	    (graph-calls function
			 (get-a-grapher-window
			   *grapher-selection-mode*
			   'stand-alone-call-graphers))))
      function *force-exposure*))

(DEFUN 4maybe-graph-function* (FUNCTION &optional (thing nil) &rest ignore)
"2Graphs Function in a grapher if it knows how to.*"
  (IF (inspect-graph-function-p function thing)
      (inspect-graph-function function thing)
      (tv:notify tv:selected-window "3Could not graph ~S.*" function)))

(DEFFLAVOR 4grapher-perspective* () (basic-perspective)
  (:documentation "3Perspectives for graphing.*"))

;1; Make sure that we know how to graph functions named by symbols.*
(defperspective 4:call-graph* (x show-x)
  :this-perspective-applicable-function
    (OR (AND (OR (SYMBOLP x) (CONSP x)) (si:fdefinition-safe x))
	(COMPILED-FUNCTION-P x))
  :menu-item-name "3Call Graph*"
  :side-effect-function
    (inspect-graph-function
      (IF (COMPILED-FUNCTION-P x) x (si:fdefinition-safe x)))
  :priority -1
  :flavor grapher-perspective)

;1-------------------------------------------------------------------------------*

(DEFUN 4edit-graphed-function* (FUNCTION)
"2Invokes Zmacs to edit the source of a function that has been graphed.*"
  (try-and-edit (coerce-to-node function)))

;1-------------------------------------------------------------------------------*
;1 Filter function definitions.*
;1-------------------------------------------------------------------------------*

(DEFUN 4same-package* (FUNCTION as child-of window)
"2Will filter out Function if it is not in the same package as As.*"
  (IGNORE child-of window)
  (IF as
      (LET ((fun (FUNCTION-NAME (the-fef function)))
	    (fas (FUNCTION-NAME (the-fef as))))
	  (NOT (AND (SYMBOLP fun) (SYMBOLP fas)
		     (EQUAL (SYMBOL-PACKAGE fun) (SYMBOL-PACKAGE fas)))))
      nil))

(DEFUN 4not-sys-functions* (FUNCTION as child-of window)
"2Will filter out Function if it is in the Sys package.*"
  (IGNORE as child-of window)
  (LET ((fun (FUNCTION-NAME (the-fef function))))
      (AND (SYMBOLP fun) (EQUAL (FIND-PACKAGE "3SYS*") (SYMBOL-PACKAGE fun)))))

(DEFUN 4not-called-by-sys-functions* (FUNCTION as child-of window)
"2Will filter out Function if it has been called by a function in the
 Sys package.*"
  (IGNORE function as window)
  (LET ((fun (FUNCTION-NAME (the-fef child-of))))
      (AND (SYMBOLP fun) (EQUAL (FIND-PACKAGE "3SYS*") (SYMBOL-PACKAGE fun)))))

(DEFUN 4not-called-by-lisp-functions* (FUNCTION as child-of window)
"2Will filter out Function if it has been called by a function in the
 Lisp package.*"
  (IGNORE function as window)
  (LET ((fun (FUNCTION-NAME (the-fef child-of))))
      (AND (SYMBOLP fun) (EQUAL (FIND-PACKAGE "3LISP*") (SYMBOL-PACKAGE fun)))))

(DEFUN 4not-%-functions* (FUNCTION as child-of window)
"2Will filter out Function if it has a name with a % in it.*"
  (IGNORE as child-of window)
  (LET ((fun (FUNCTION-NAME (the-fef function))))
      (AND (SYMBOLP fun) (SEARCH "3%*" (SYMBOL-NAME fun) :test #'STRING-EQUAL))))

(DEFUN 4not-*-functions* (FUNCTION as child-of window)
"2Will filter out Function if it has a name with a % in it.  Such functions are
 often generated by compiler optimisers for system functions.*"
  (IGNORE as child-of window)
  (LET ((fun (FUNCTION-NAME (the-fef function))))
      (AND (SYMBOLP fun)
	    (SEARCH "3**" (SYMBOL-NAME fun) :test #'STRING-EQUAL)
	    (> (LENGTH (THE string (SYMBOL-NAME fun))) 1))))

(DEFUN 4only-methods-below-generic-functions* (FUNCTION as child-of window)
"2Filters out Function if it is called by a generic function but is not a method
 of that generic function.*"
  (IGNORE as window)
  (AND (generic-function-p-safe child-of)
       (LET ((name (FUNCTION-NAME (the-fef function))))
	   (NOT (AND (CONSP name) (EQUAL 'ticlos:method (FIRST name)))))))

(DEFUN 4not-methods-below-generic-functions* (FUNCTION as child-of window)
"2Filters out Function if it is a method on a generic function but is not
 called directly by it.*"
  (IGNORE as window)
  (AND (generic-function-p-safe child-of)
       (LET ((name (FUNCTION-NAME (the-fef function))))
	   (AND (CONSP name) (EQUAL 'ticlos:method (FIRST name))))))

(DEFUN 4same-source-file* (FUNCTION as child-of window)
"2Filters out Filter unless it is defined in the same source file as As.*"
  (IGNORE child-of window)
  (IF as
      (LET ((fun (FUNCTION-NAME (the-fef function)))
	    (fas (FUNCTION-NAME (the-fef as))))
	  (NOT (AND (SYMBOLP fun) (SYMBOLP fas)
		     (GET fun :source-file-name)
		     (GET fas :source-file-name)
		     (fs::pathname-equal
		       (fs:default-pathname (path-string fun 'DEFUN))
		       (fs:default-pathname (path-string fas 'DEFUN))))))
      nil))

(DEFUN 4no-macros* (FUNCTION as child-of window)
"2Filters out Function if it is a macro.*"
  (IGNORE as child-of window)
  (AND (MACRO-FUNCTION (FUNCTION-NAME (the-fef function)))
        (NOT (SPECIAL-FORM-P (FUNCTION-NAME (the-fef function))))))

(DEFUN 4no-special-forms* (FUNCTION as child-of window)
"2Filters out Function if it is a special form.*"
  (IGNORE as child-of window)
  (SPECIAL-FORM-P (FUNCTION-NAME (the-fef function))))

(DEFUN 4not-called-by-macros* (FUNCTION as child-of window)
"2Will filter out Function if it has been called by a macro.*"
  (IGNORE function as window)
  (AND (MACRO-FUNCTION (FUNCTION-NAME (the-fef child-of)))
        (NOT (SPECIAL-FORM-P (FUNCTION-NAME (the-fef child-of))))))

(DEFUN 4not-called-by-special-forms* (FUNCTION as child-of window)
"2Will filter out Function if it has been called by a special form.*"
  (IGNORE function as window)
  (AND (SYMBOLP (FUNCTION-NAME (the-fef child-of)))
        (SPECIAL-FORM-P (FUNCTION-NAME (the-fef child-of)))))

(DEFUN 4not-called-by-interpretted-functions* (FUNCTION as child-of window)
"2Will filter out Function if it has been called by an interpretted function.*"
  (IGNORE function as window)
  (EQUAL child-of (the-fef child-of)))

(DEFUN 4select-grapher-filters* (FUNCTION window &rest ignore)
"2Is called when the user right buttons on a function vertex.  Pops up a menu
 olf filters to select.  If the user selects any then these are recorded, as is
 the currently selected function.*"
  (MULTIPLE-VALUE-BIND (choices chose-p)
        (w:multiple-menu-choose *filter-items*
	  :label (FORMAT nil "3Filter Graphing for ~S*"
			  (FUNCTION-NAME (the-fef function)))
	  :highlighted-items
	  (REMOVE-IF-NOT
	    #'(lambda (x) (MEMBER (THIRD x) *selected-filters*))
	    *filter-items*))
    (IF chose-p
        (PROGN (SETQ *as-function* (the-fef function))
	        (SETQ *selected-filters* choices)
		(SEND window :set-graph-behaviour-changed t))
       nil)))

(DEFUN 4set-graph-calls-variables* (FUNCTION)
"2Sets up interesting variables to do with the function that has been
 clicked on.*"
  (IGNORE function))

;1-------------------------------------------------------------------------------*

(DEFUN 4graph-something* (name)
"2Is called when the user has selected the Graph command from some tool.  It
 tries to find a way to graph Name and, if it can find such a thing, will
 graph it.  If there are multiple options than the user will be prompted with
 a menu.*"
  (MULTIPLE-VALUE-BIND (object inspect-p)
      (map-into-show-x name nil 'grapher-perspective)
    (IF (AND object (NOT inspect-p))
       (FORMAT *query-io* "3~&Graphed ~S.*" object)
       (FORMAT *query-io* "3~&~S not graphed.*" name))))

zwei:
(defcom 4zwei::com-graph*
"3Prompt for something and graphs it.*" ()
  (LET ((fcn (zwei::read-function-name "3Graph*"
				 (relevant-function-name (point) nil t t t)
				 'zwei::aarray-ok 'zwei::multiple-ok))
	(*print-case* :capitalize))
       (tv::graph-something fcn))
  dis-none)

;1;; Record the new zmacs commands.*
(zwei::set-comtab zwei::*standard-comtab* '(#\c-sh-g zwei::com-graph)
		 (zwei::make-command-alist '(zwei::com-graph)))

(zwei::set-comtab zwei::*standard-comtab* '(#\m-sh-g zwei::com-graph)
		 (zwei::make-command-alist '(zwei::com-graph)))

(DEFCOMMAND 4graph-cmd* nil			
  '(:description "3Graph something.*"
    :names ("3Graph*")
    :keys (#\m-sh-g #\c-sh-g))
   (DECLARE (SPECIAL user history = inspectors frame))
   (SEND user :clear-screen)
   (FORMAT user "3~&Something to graph:*")
   (MULTIPLE-VALUE-BIND (value punt-p)
       (inspect-get-value-from-user user history inspectors)
     (OR punt-p (graph-something value)))
   (SEND frame :handle-prompt))

eh:
(DEFCOMMAND 4eh::comw-graph* ()
            '(:description "3Graph something.*"
              :names "3Graph*"
	      :keys (#\m-sh-g #\c-sh-g)) 
   (SEND *window-debugger* :set-who-line-doc-string-overide
	 "3Select something to graph.*")
   (UNWIND-PROTECT
       (tv::graph-something (window-read-thing "3to graph*"))
      (SEND *window-debugger* :set-who-line-doc-string-overide nil)
      (SEND *window-debugger* :handle-prompt)))

;1 TAC 08-08-89 - this is now a parameter in DTCE*
;1; This is defined as a constant in DTCE.*
;1(setf (get '*All-Consistancy-Commands* 'compiler:system-constant) nil)*

(SETQ *all-consistancy-commands* nil) 

;1; Record Graph as a consistency command.*
(PUSHNEW '("3Graph*" :value
	     (((#\c-sh-g zwei::com-graph)
	       (#\m-sh-g zwei::com-graph))
	      ((graph-cmd nil))
	      ((eh::comw-graph nil))))
	    *all-consistancy-commands*
	    :test #'EQUALP)

;1; Reinstall so that we can see Graph as a command.*
(install-consistency-commands)

(reinstall-commands 'general-inspector)

;1-------------------------------------------------------------------------------*

(DEFUN 4the-class* (the-class)
"2Coerces The-Class into a class or a flavor object.*"
  (TYPECASE the-class
    (symbol (the-class (OR (class-named-safe the-class t)
			   (GET the-class 'si::flavor)
			   (FERROR nil "3~S is not a class.*" the-class))))
    (otherwise (IF (OR (class-p-safe the-class)
		       (TYPEP the-class 'si::flavor))
		   the-class
		   (IF (any-sort-of-clos-instance-p the-class)
		       (class-named-safe (TYPE-OF the-class))
		       (IF (INSTANCEP the-class)
			   (TYPE-OF the-class)
			   (FERROR nil "3~S is not a class.*" the-class)))))))

(DEFUN 4class-kids* (class)
"2Returns a list of the direct superclasses or local component flavors of class.*"
  (IF (class-p-safe (coerce-to-node class))
      (class-local-supers-safe (coerce-to-node class))
      (IF (TYPEP (coerce-to-node class) 'si::flavor)
	  (MAPCAR #'(lambda (x) (GET x 'si::flavor))
		  (si::flavor-depends-on (coerce-to-node class)))
	  nil)))

(DEFUN 4class-print-fun* (class)
"2Prints out Class neatly as a graph node.*" 
  (LET ((the-class (coerce-to-node class))
        (*package* nil))
      (FORMAT nil "3~S~A*"
	      (IF (class-p-safe the-class)
		  (class-name-safe the-class)
		  (IF (TYPEP the-class 'si::flavor)
		      (si::flavor-name the-class)
		      the-class))
	      (more-kids-indicator
		(IF (AND (TYPEP class 'vertex-node)
			 (vertex-node-vertex class))
		    (AND (NOT (SEND (vertex-node-vertex class) :child-edges))
			 (class-kids (coerce-to-node the-class)))
		    (class-kids the-class))))))

(DEFUN 4edit-graphed-thing* (thing)
"2Edits the source of a thing that has been graphed.*"
  (try-and-edit (coerce-to-node thing)))

(DEFUN 4inspect-a-graphed-class* (class)
  (INSPECT (allocate-data
	     (IF (class-p-safe class)
		 'show-clos-class
		 'show-flavor)
	     (coerce-to-node class))))

(DEFVAR 4*graph-class-item-type-alist**
  '((:vertex graph-class
     "3L: Graph this Class; M: Drag object; Sh-M/M2: Overview; R: Menu of operations*"
     ("3Inspect*" :value inspect-a-graphed-class :documentation
      "3Inspect this class.*")
     ("3Edit*" :value edit-graphed-thing :documentation "3Edit this class.*")))
"2The item type AList for the class grapher.  New menu items should be added to
 the end.*")

(DEFVAR 4*default-class-graph-depth** nil
"2The default depth of class graphs.*")

(DEFFLAVOR 4class-vertex* () (boxed-filled-vertex)
  (:documentation
    "3A type of vertex for class graphs.  Classes will be filled flavors won't.*"))

(DEFMETHOD 4(class-vertex :after :init*) (IGNORE)
"2Makes sure that each vertex in the graph is set up so that it's box/fill
 status is right.  Classes are filled.  Anything else is boxed.*"
  (LET ((class (the-class (coerce-to-node item))))
      (LET ((class-p (class-p-safe class)))
	  (IF (AND class-p
		    (class-name-safe class)
		    (SYMBOLP (class-name-safe class))
		    (NOT (GET (class-name-safe class) 'si::flavor)))
	     (PROGN (SET-IN-INSTANCE self 'fill-myself-p t)
		    (SET-IN-INSTANCE self 'box-myself-p nil))
	     (PROGN (SET-IN-INSTANCE self 'fill-myself-p nil)
		    (SET-IN-INSTANCE self 'box-myself-p t))))))

(DEFUN 4graph-class* (class &optional (on :from-resource) &rest ignore)
"2Graphs Class, which is a class or a flavor on the window On.  If On is
 defaulted then a window is selected from a resource.  The returned value is
 the window that was used.*"
  (LET ((class (the-class class)))
       (plot-a-graph class
	  :on on
	  :child-function 'class-kids
	  :print-function 'class-print-fun
	  :vertex-flavor 'class-vertex
	  :depth *default-class-graph-depth*
	  :tree-p t
	  :item-type-alist *graph-class-item-type-alist*
	  :dont-graph-if-already-graphing-this-node t
	  :label
	    (IF (class-p-safe class)
		(FORMAT nil "3~A's superclasses*" (class-name-safe class))
		(FORMAT nil "3~A's components*"
			(TYPECASE class
			  (si::flavor (si::flavor-name class))
			  (symbol (si::flavor-name (GET class 'si::flavor)))
			  (otherwise (FERROR "3~S is not a class or flavor.*"
					     class))))))))

(DEFUN 4inspect-graph-class* (class)
"2Graphs a class from the inspector.  Spins off a process to do it so
 that we don't have selection/exposure problems.*"
  (PROCESS-RUN-FUNCTION
    '(:name "3Expose Grapher*" :priority -1)
    #'(lambda (class expose-p)
	(LET ((*force-exposure* expose-p))
	    (DECLARE (SPECIAL *force-exposure*))
	    (graph-class class (get-a-grapher-window))))
    class
    *force-exposure*))

(DEFUN 4inspect-graph-class-p* (class &optional (thing nil))
"2Is true if Class is a class or flavor that we can graph, be it a class or
 the name of one.*"
  (IGNORE thing)
  (OR (AND (SYMBOLP class) (class-named-safe class t))
      (AND (SYMBOLP class) (GET class 'si::flavor))
      (class-p-safe class)
      (TYPEP class 'si::flavor)))

;1; Remember that we can graph classes.*

(defperspective 4:inheritance-graph* (x show-x)
  :this-perspective-applicable-function (inspect-graph-class-p x)
  :menu-item-name "3Inheritance Graph*"
  :side-effect-function (inspect-graph-class x)
  :priority -1
  :flavor grapher-perspective)

;1-------------------------------------------------------------------------------*
;1 TAC 08-17-89 - not going to wire meta-. to the grapher for possibility editting.*
;1;; Maybe wire M-. to the grapher.*

;1(DEFVAR zwei:*wire-grapher-to-m-.* nil*
;1"When true the grapher will be invoked whenever zmacs goes to a new possibility*
;1 if it can be.")*

;1;---------------------------------------------------------------------------------------------*
;1(LET ((compiler:compile-encapsulations-flag t))*
;1  (ADVISE zwei:execute-possibility :around :wire-to-grapher nil*
;1    (LOCALLY (DECLARE (SPECIAL *inside-execute-possibility*))*
;	1     (LET ((results (LET-IF (NOT (BOUNDP '*inside-execute-possibility*))*
;				1    ((*inside-execute-possibility* t))*
;			1      ;; Bind *inside-execute-possibility* so that we only graph once.  *
;			1      ;; Zwei:Execute-Possibility is called recursively.*
;			1      (MULTIPLE-VALUE-LIST :do-it))))*
;	1       (IF (OR (BOUNDP '*inside-execute-possibility*)*
;		1       (NOT zwei:*wire-grapher-to-m-.*)) ;; var is above *
;		1   nil*
;		1   (LET ((*force-exposure* nil)) ;; var is in funtional-interface*
;		1     (DECLARE (SPECIAL *force-exposure*))*
;		1     (map-into-show-x (SECOND (SECOND arglist)) nil*
;				1      'grapher-perspective)))*
;	1       (VALUES-LIST results)))))*
;1;---------------------------------------------------------------------------------------------*

;1;---------------------------------------------------------------------------------------------*

;1(LET ((compiler:compile-encapsulations-flag t))*
;1  (ADVISE zwei:edit-definition-1 :around :wire-to-grapher nil*
;1    (LOCALLY (DECLARE (SPECIAL *inside-execute-possibility*))*
;	1     (LET ((results (LET-IF (NOT (BOUNDP '*inside-execute-possibility*))*
;				1    ((*inside-execute-possibility* t))*
;			1            ;; Bind *inside-execute-possibility* so that we only graph once.  *
;			1            ;; Zwei:Execute-Possibility is called recursively.*
;			1      (MULTIPLE-VALUE-LIST :do-it))))*
;	1       (IF (OR (BOUNDP '*inside-execute-possibility*)*
;		1       (NOT zwei:*wire-grapher-to-m-.*)) ;; var is above *
;		1   nil*
;		1   (LET ((*force-exposure* nil))*
;		1     (DECLARE (SPECIAL *force-exposure*)) ;; var is in funtional-interface*
;		1     (map-into-show-x (FIRST arglist) nil*
;				1      'grapher-perspective)))*
;	1       (VALUES-LIST results)))))*
;1;---------------------------------------------------------------------------------------------
